//  $Id: fmkey.cc,v 1.2 2001/07/13 15:15:06 nishi Exp $
//
//  Copyright (C) 2001 Shouhei Nishi.
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA


// Now features proper implementation of keyboard opcodes 0xF4 to 0xF6
// Silently ignores PS/2 keyboard extensions (0xF7 to 0xFD)
// Explicit panic on resend (0xFE)
//
// Emmanuel Marty <core@ggi-project.org>


#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS  bx_keyboard.


#define VERBOSE_KBD_DEBUG 0


#define MOUSE_MODE_RESET  10
#define MOUSE_MODE_STREAM 11
#define MOUSE_MODE_REMOTE 12
#define MOUSE_MODE_WRAP   13

bx_keyb_c bx_keyboard;

#if BX_USE_KEY_SMF
#define this (&bx_keyboard)
#endif



bx_keyb_c::bx_keyb_c(void)
{
  // constructor
  // should zero out state info here???
  memset( &s, 0, sizeof(s) );
}

bx_keyb_c::~bx_keyb_c(void)
{
  // destructor
  BX_DEBUG(("Exit.\n"));
}


// flush internal buffer and reset keyboard settings to power-up condition
  void
bx_keyb_c::resetinternals(Boolean powerup)
{
  Bit32u   i;

  BX_KEY_THIS s.kbd_internal_buffer.num_elements = 0;
  for (i=0; i<BX_KBD_ELEMENTS; i++)
    BX_KEY_THIS s.kbd_internal_buffer.buffer[i] = 0;
  BX_KEY_THIS s.kbd_internal_buffer.head = 0;

  if (powerup) {
    BX_KEY_THIS s.kbd_internal_buffer.delay = 1; // 500 mS
    BX_KEY_THIS s.kbd_internal_buffer.repeat_rate = 0x0b; // 10.9 chars/sec
    }
}



  void
bx_keyb_c::init(bx_devices_c *d, bx_cmos_c *cmos)
{
  BX_KEY_THIS setprefix("[KBD ]");
  BX_KEY_THIS settype(KBDLOG);
  Bit32u   i;

  BX_KEY_THIS devices = d;

  BX_KEY_THIS devices->register_irq(1, "8042 Keyboard controller");

  BX_KEY_THIS devices->register_io_read_handler(this, read_handler,
                                      0x0600, "8042 Keyboard controller");
  BX_KEY_THIS devices->register_io_read_handler(this, read_handler,
                                      0x0602, "8042 Keyboard controller");
  BX_KEY_THIS devices->register_io_read_handler(this, read_handler,
                                      0x0604, "8042 Keyboard controller");
  BX_KEY_THIS devices->register_io_write_handler(this, write_handler,
                                      0x0600, "8042 Keyboard controller");
  BX_KEY_THIS devices->register_io_write_handler(this, write_handler,
                                      0x0602, "8042 Keyboard controller");
  BX_KEY_THIS devices->register_io_write_handler(this, write_handler,
                                      0x0604, "8042 Keyboard controller");

  resetinternals(1);

#if 0
  BX_KEY_THIS s.mouse_internal_buffer.num_elements = 0;
  for (i=0; i<BX_MOUSE_BUFF_SIZE; i++)
    BX_KEY_THIS s.mouse_internal_buffer.buffer[i] = 0;
  BX_KEY_THIS s.mouse_internal_buffer.head = 0;
#endif

  BX_KEY_THIS s.kbd_controller.ST7 = 0;
  BX_KEY_THIS s.kbd_controller.ST6  = 0;
  BX_KEY_THIS s.kbd_controller.ST5 = 0;
  BX_KEY_THIS s.kbd_controller.ST4 = 1;
  BX_KEY_THIS s.kbd_controller.c_d  = 1;
  BX_KEY_THIS s.kbd_controller.sysf = 0;
  BX_KEY_THIS s.kbd_controller.inpb = 0;
  BX_KEY_THIS s.kbd_controller.outb = 0;

  BX_KEY_THIS s.kbd_controller.allow_irq1 = 0;
  BX_KEY_THIS s.kbd_controller.kbd_output_buffer = 0;
  //BX_KEY_THIS s.kbd_controller.last_comm = 0;
  BX_KEY_THIS s.kbd_controller.expecting_port600h = 0;
  //BX_KEY_THIS s.kbd_controller.aux_output_buffer = 0;
  BX_KEY_THIS s.kbd_controller.irq1_requested = 0;

//BX_DEBUG(( "# keyboard_serial_delay is %u usec\n",
//        (unsigned) bx_options.keyboard_serial_delay));
  BX_KEY_THIS s.kbd_controller.timer_pending = 0;

#if 0
  // Mouse initialization stuff
  BX_KEY_THIS s.mouse.sample_rate     = 100; // reports per second
  BX_KEY_THIS s.mouse.resolution_cpmm = 4;   // 4 counts per millimeter
  BX_KEY_THIS s.mouse.scaling         = 1;   /* 1:1 (default) */
  BX_KEY_THIS s.mouse.mode            = MOUSE_MODE_RESET;
  BX_KEY_THIS s.mouse.enable          = 0;
#endif

  for (i=0; i<BX_KBD_CONTROLLER_QSIZE; i++)
    BX_KEY_THIS s.controller_Q[i] = 0;
  BX_KEY_THIS s.controller_Qsize = 0;
  BX_KEY_THIS s.controller_Qsource = 0;

  BX_DEBUG(("Init.\n"));
}

#define RETURN(x) do { ret = (x); goto read_return; } while (0)

  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_keyb_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_KEY_SMF
  bx_keyb_c *class_ptr = (bx_keyb_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}


  Bit32u
bx_keyb_c::read(Bit32u   address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_KEY_SMF

  Bit32u ret = 0;

  if (io_len > 1)
    BX_PANIC(("kbd: io read to address %08x, len=%u\n",
             (unsigned) address, (unsigned) io_len));


//BX_DEBUG(( "read from port 0x%04x\n", (unsigned) address));

  if (address == 0x600) { /* output buffer */
    Bit8u   val;
#if 0
    if (BX_KEY_THIS s.kbd_controller.auxb) { /* mouse byte available */
      val = BX_KEY_THIS s.kbd_controller.aux_output_buffer;
      BX_KEY_THIS s.kbd_controller.aux_output_buffer = 0;
      BX_KEY_THIS s.kbd_controller.outb = 0;
      BX_KEY_THIS s.kbd_controller.auxb = 0;

      if (BX_KEY_THIS s.controller_Qsize) {
        unsigned i;
        BX_KEY_THIS s.kbd_controller.aux_output_buffer = BX_KEY_THIS s.controller_Q[0];
        BX_KEY_THIS s.kbd_controller.outb = 1;
        BX_KEY_THIS s.kbd_controller.auxb = 1;
        if (BX_KEY_THIS s.kbd_controller.allow_irq12)
          BX_KEY_THIS s.kbd_controller.irq12_requested = 1;
        for (i=0; i<BX_KEY_THIS s.controller_Qsize-1; i++) {
          // move Q elements towards head of queue by one
          BX_KEY_THIS s.controller_Q[i] = BX_KEY_THIS s.controller_Q[i+1];
          }
        BX_KEY_THIS s.controller_Qsize--;
        }

//BX_INFO(("mouse: ___io_read aux = 0x%02x\n", (unsigned) val));

      activate_timer();
      if (bx_dbg.keyboard)
        BX_INFO(("READ(%02x) = %02x\n", (unsigned) address,
          (unsigned) val));
      RETURN(val);
      }
    else
#endif 
    if (BX_KEY_THIS s.kbd_controller.outb) { /* kbd byte available */
      val = BX_KEY_THIS s.kbd_controller.kbd_output_buffer;
      BX_KEY_THIS s.kbd_controller.kbd_output_buffer = 0;
      BX_KEY_THIS s.kbd_controller.outb = 0;
//BX_DEBUG(( "___io_read kbd\n"));

      if (BX_KEY_THIS s.controller_Qsize) {
        unsigned i;
        BX_KEY_THIS s.kbd_controller.kbd_output_buffer = BX_KEY_THIS s.controller_Q[0];
        BX_KEY_THIS s.kbd_controller.outb = 1;
        if (BX_KEY_THIS s.kbd_controller.allow_irq1)
          BX_KEY_THIS s.kbd_controller.irq1_requested = 1;
        for (i=0; i<BX_KEY_THIS s.controller_Qsize-1; i++) {
          // move Q elements towards head of queue by one
          BX_KEY_THIS s.controller_Q[i] = BX_KEY_THIS s.controller_Q[i+1];
          }
        BX_KEY_THIS s.controller_Qsize--;
        }

      activate_timer();
      if (bx_dbg.keyboard)
        BX_INFO(("READ(%02x) = %02x\n", (unsigned) address,
          (unsigned) val));
      RETURN(val);
      }
    else {
      if (bx_dbg.keyboard) {
        BX_INFO(("num_elements = %d\n", BX_KEY_THIS s.kbd_internal_buffer.num_elements));
        BX_INFO(("read from port 60h with outb empty\n"));
        }
      val = 0;
      RETURN(val);
      }
    }

  else if (address == 0x602) { /* status register */
    Bit8u   val;

    val = (BX_KEY_THIS s.kbd_controller.ST7  << 7)  |
          (BX_KEY_THIS s.kbd_controller.ST6  << 6)  |
          (BX_KEY_THIS s.kbd_controller.ST5  << 5)  |
          (BX_KEY_THIS s.kbd_controller.ST4  << 4)  |
          (BX_KEY_THIS s.kbd_controller.c_d  << 3)  |
          (BX_KEY_THIS s.kbd_controller.sysf << 2)  |
          (BX_KEY_THIS s.kbd_controller.inpb << 1)  |
          BX_KEY_THIS s.kbd_controller.outb;
    RETURN(val);
    }

  else {
    BX_PANIC(("KBD: unknown address in io read to keyboard port %x\n",
      (unsigned) address));
    RETURN(0); /* keep compiler happy */
    }

  read_return:
  if (bx_dbg.keyboard)
	BX_INFO(("keyboard: 8-bit read from %04x = %02x\n", (unsigned)address, ret));
  return ret;
}


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_keyb_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_KEY_SMF
  bx_keyb_c *class_ptr = (bx_keyb_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}

  void
bx_keyb_c::write( Bit32u   address, Bit32u   value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_KEY_SMF
  //Bit8u   command_byte;

  if (io_len > 1)
    BX_PANIC(("kbd: io write to address %08x, len=%u\n",
             (unsigned) address, (unsigned) io_len));

  if (bx_dbg.keyboard)
	BX_INFO(("keyboard: 8-bit write to %04x = %02x\n", (unsigned)address, (unsigned)value));


//BX_DEBUG(("WRITE(%02x) = %02x\n", (unsigned) address,
//      (unsigned) value));

  switch (address) {
    case 0x600: // input buffer
      // if expecting data byte from command last sent to port 64h
      if (BX_KEY_THIS s.kbd_controller.expecting_port600h) {
        BX_KEY_THIS s.kbd_controller.expecting_port600h = 0;
        // data byte written last to 0x60
        BX_KEY_THIS s.kbd_controller.c_d = 0;
        if (BX_KEY_THIS s.kbd_controller.inpb) {
          BX_PANIC(("write to port 60h, not ready for write\n"));
          }
        switch (BX_KEY_THIS s.kbd_controller.last_comm) {
	}
      }
      else {
        // data byte written last to 0x60
        BX_KEY_THIS s.kbd_controller.c_d = 0;
        BX_KEY_THIS s.kbd_controller.expecting_port600h = 0;
        //kbd_ctrl_to_kbd(value);
        }
      break;

    case 0x602: // control register
      // command byte written last to 0x64
      BX_KEY_THIS s.kbd_controller.c_d = 1;
      BX_KEY_THIS s.kbd_controller.last_comm = value;
      // most commands NOT expecting port60 write next
      BX_KEY_THIS s.kbd_controller.expecting_port600h = 0;

      switch (value) {
        default:
          BX_INFO(("KBD: unsupported io write to keyboard port %x, value = %x\n",
            (unsigned) address, (unsigned) value));
          break;
        }
      break;

    case 0x604:
      BX_KEY_THIS s.kbd_controller.allow_irq1 = ((value & 1) != 0);
      break;

    default: BX_PANIC(("KBD: unknown address in bx_keyb_c::write()\n"));
    }
}


  void
bx_keyb_c::gen_scancode(Bit32u   key)
{
  Bit8u   scancode;
  int extended;

  BX_DEBUG(( "gen_scancode %lld %x\n", bx_pc_system.time_ticks(), key));

  if (bx_dbg.keyboard)
    BX_INFO(("KBD: gen_scancode(): scancode: %08x\n", (unsigned) key));

  // should deal with conversions from KSCAN to system scan codes here

  extended = 0;
  switch (key & 0xff) {
    case BX_KEY_CTRL_L:  scancode = 0x1d; break;
    case BX_KEY_CTRL_R:  extended = 1; scancode = 0x1d; break;
    case BX_KEY_SHIFT_L: scancode = 0x2a; break;
    case BX_KEY_SHIFT_R: scancode = 0x36; break;
    case BX_KEY_ESC:   scancode = 0x01; break;

    case BX_KEY_ALT_L: scancode = 0x38; break;
    case BX_KEY_ALT_R: extended = 1; scancode = 0x38; break;

    case BX_KEY_A:     scancode = 0x1e; break;
    case BX_KEY_B:     scancode = 0x30; break;
    case BX_KEY_C:     scancode = 0x2e; break;
    case BX_KEY_D:     scancode = 0x20; break;
    case BX_KEY_E:     scancode = 0x12; break;
    case BX_KEY_F:     scancode = 0x21; break;
    case BX_KEY_G:     scancode = 0x22; break;
    case BX_KEY_H:     scancode = 0x23; break;
    case BX_KEY_I:     scancode = 0x17; break;
    case BX_KEY_J:     scancode = 0x24; break;
    case BX_KEY_K:     scancode = 0x25; break;
    case BX_KEY_L:     scancode = 0x26; break;
    case BX_KEY_M:     scancode = 0x32; break;
    case BX_KEY_N:     scancode = 0x31; break;
    case BX_KEY_O:     scancode = 0x18; break;
    case BX_KEY_P:     scancode = 0x19; break;
    case BX_KEY_Q:     scancode = 0x10; break;
    case BX_KEY_R:     scancode = 0x13; break;
    case BX_KEY_S:     scancode = 0x1f; break;
    case BX_KEY_T:     scancode = 0x14; break;
    case BX_KEY_U:     scancode = 0x16; break;
    case BX_KEY_V:     scancode = 0x2f; break;
    case BX_KEY_W:     scancode = 0x11; break;
    case BX_KEY_X:     scancode = 0x2d; break;
    case BX_KEY_Y:     scancode = 0x15; break;
    case BX_KEY_Z:     scancode = 0x2c; break;

    case BX_KEY_0:     scancode = 0x0b; break;
    case BX_KEY_1:     scancode = 0x02; break;
    case BX_KEY_2:     scancode = 0x03; break;
    case BX_KEY_3:     scancode = 0x04; break;
    case BX_KEY_4:     scancode = 0x05; break;
    case BX_KEY_5:     scancode = 0x06; break;
    case BX_KEY_6:     scancode = 0x07; break;
    case BX_KEY_7:     scancode = 0x08; break;
    case BX_KEY_8:     scancode = 0x09; break;
    case BX_KEY_9:     scancode = 0x0a; break;

    case BX_KEY_SPACE:        scancode = 0x39; break;
    case BX_KEY_SINGLE_QUOTE: scancode = 0x28; break;
    case BX_KEY_COMMA:        scancode = 0x33; break;
    case BX_KEY_PERIOD:       scancode = 0x34; break;
    case BX_KEY_SLASH:        scancode = 0x35; break;

    case BX_KEY_SEMICOLON:     scancode = 0x27; break;
    case BX_KEY_EQUALS:        scancode = 0x0d; break;

    case BX_KEY_LEFT_BRACKET:  scancode = 0x1a; break;
    case BX_KEY_BACKSLASH:     scancode = 0x2b; break;
    case BX_KEY_RIGHT_BRACKET: scancode = 0x1b; break;
    case BX_KEY_MINUS:         scancode = 0x0c; break;
    case BX_KEY_GRAVE:         scancode = 0x29; break;

    case BX_KEY_BACKSPACE:     scancode = 0x0e; break;
    case BX_KEY_KP_ENTER:
    case BX_KEY_ENTER:         scancode = 0x1c; break;
    case BX_KEY_TAB:           scancode = 0x0f; break;

    case BX_KEY_LEFT:          extended = 1;
    case BX_KEY_KP_LEFT:       scancode = 0x4b; break;
    case BX_KEY_RIGHT:         extended = 1;
    case BX_KEY_KP_RIGHT:      scancode = 0x4d; break;
    case BX_KEY_UP:            extended = 1;
    case BX_KEY_KP_UP:         scancode = 0x48; break;
    case BX_KEY_DOWN:          extended = 1;
    case BX_KEY_KP_DOWN:       scancode = 0x50; break;

    case BX_KEY_INSERT:        extended = 1;
    case BX_KEY_KP_INSERT:        scancode = 0x52; break;
    case BX_KEY_DELETE:        extended = 1;
    case BX_KEY_KP_DELETE:        scancode = 0x53; break;
    case BX_KEY_HOME:          extended = 1;
    case BX_KEY_KP_HOME:          scancode = 0x47; break;
    case BX_KEY_END:           extended = 1;
    case BX_KEY_KP_END:           scancode = 0x4f; break;
    case BX_KEY_PAGE_UP:       extended = 1;
    case BX_KEY_KP_PAGE_UP:       scancode = 0x49; break;
    case BX_KEY_PAGE_DOWN:     extended = 1;
    case BX_KEY_KP_PAGE_DOWN:     scancode = 0x51; break;

    case BX_KEY_KP_ADD:           scancode = 0x4e; break;
    case BX_KEY_KP_SUBTRACT:      scancode = 0x4a; break;
    case BX_KEY_KP_5:             scancode = 0x4c; break;
    case BX_KEY_KP_MULTIPLY:
      BX_DEBUG(( "Grey Multiply key not on 83-key keyboard\n" ));
      return;
    case BX_KEY_KP_DIVIDE:
      BX_DEBUG(( "Grey Divide key not on 83-key keyboard\n" ));
      return;
    case BX_KEY_NUM_LOCK:         scancode = 0x45; break;

    case BX_KEY_F1:               scancode = 0x3b; break;
    case BX_KEY_F2:               scancode = 0x3c; break;
    case BX_KEY_F3:               scancode = 0x3d; break;
    case BX_KEY_F4:               scancode = 0x3e; break;
    case BX_KEY_F5:               scancode = 0x3f; break;
    case BX_KEY_F6:               scancode = 0x40; break;
    case BX_KEY_F7:               scancode = 0x41; break;
    case BX_KEY_F8:               scancode = 0x42; break;
    case BX_KEY_F9:               scancode = 0x43; break;
    case BX_KEY_F10:              scancode = 0x44; break;
    case BX_KEY_F11:              scancode = 0x57; break;
    case BX_KEY_F12:              scancode = 0x58; break;

    default:
      BX_DEBUG(( "bx_keyb_c::gen_scancode : Unhandled %u\n",
        (unsigned) key));
      return;
    }
  if (extended) kbd_enQ(0xE0);
  if (key & BX_KEY_RELEASED)
    scancode |= 0x80;
  kbd_enQ(scancode);
}



  void
bx_keyb_c::controller_enQ(Bit8u   data, unsigned source)
{
  // source is 0 for keyboard, 1 for mouse

  if (bx_dbg.keyboard)
    BX_INFO(("controller_enQ(%02x)\n", (unsigned) data));

  if (BX_KEY_THIS s.kbd_controller.outb)
    BX_INFO(("controller_enQ(): OUTB set!\n"));

  // see if we need to Q this byte from the controller
  if (BX_KEY_THIS s.kbd_controller.outb) {
    if (BX_KEY_THIS s.controller_Qsize >= BX_KBD_CONTROLLER_QSIZE)
      BX_PANIC(("controller_enq(): controller_Q full!\n"));
    BX_KEY_THIS s.controller_Q[BX_KEY_THIS s.controller_Qsize++] = data;
    BX_KEY_THIS s.controller_Qsource = source;
    return;
    }

  if (source == 0) { // keyboard
    BX_KEY_THIS s.kbd_controller.kbd_output_buffer = data;
    BX_KEY_THIS s.kbd_controller.outb = 1;
    //BX_KEY_THIS s.kbd_controller.auxb = 0;
    BX_KEY_THIS s.kbd_controller.inpb = 0;
    if (BX_KEY_THIS s.kbd_controller.allow_irq1)
      BX_KEY_THIS s.kbd_controller.irq1_requested = 1;
    }
#if 0
  else { // mouse
    BX_KEY_THIS s.kbd_controller.aux_output_buffer = data;
    BX_KEY_THIS s.kbd_controller.outb = 1;
    BX_KEY_THIS s.kbd_controller.auxb = 1;
    BX_KEY_THIS s.kbd_controller.inpb = 0;
    if (BX_KEY_THIS s.kbd_controller.allow_irq12)
      BX_KEY_THIS s.kbd_controller.irq12_requested = 1;
    }
#endif
}

void
bx_keyb_c::kbd_enQ_imm(Bit8u val)
{
      int tail;

      if (BX_KEY_THIS s.kbd_internal_buffer.num_elements >= BX_KBD_ELEMENTS) {
	    BX_PANIC(("internal keyboard buffer full (imm)\n"));
	    return;
      }

      /* enqueue scancode in multibyte internal keyboard buffer */
      tail = (BX_KEY_THIS s.kbd_internal_buffer.head + BX_KEY_THIS s.kbd_internal_buffer.num_elements) %
	    BX_KBD_ELEMENTS;

      BX_KEY_THIS s.kbd_controller.kbd_output_buffer = val;
      BX_KEY_THIS s.kbd_controller.outb = 1;

      if (BX_KEY_THIS s.kbd_controller.allow_irq1)
	    BX_KEY_THIS s.kbd_controller.irq1_requested = 1;
}


  void
bx_keyb_c::kbd_enQ(Bit8u   scancode)
{
  int tail;

  if (bx_dbg.keyboard)
    BX_INFO(("enQ(%02x)\n", (unsigned) scancode));

  if (BX_KEY_THIS s.kbd_internal_buffer.num_elements >= BX_KBD_ELEMENTS) {
    BX_INFO(("internal keyboard buffer full, ignoring scancode.(%02x)\n",
      (unsigned) scancode));
    return;
    }

  /* enqueue scancode in multibyte internal keyboard buffer */
  if (bx_dbg.keyboard)
    BX_INFO(("enQ: putting scancode %02x in internal buffer\n",
      (unsigned) scancode));
  tail = (BX_KEY_THIS s.kbd_internal_buffer.head + BX_KEY_THIS s.kbd_internal_buffer.num_elements) %
   BX_KBD_ELEMENTS;
  BX_KEY_THIS s.kbd_internal_buffer.buffer[tail] = scancode;
  BX_KEY_THIS s.kbd_internal_buffer.num_elements++;

  if (!BX_KEY_THIS s.kbd_controller.outb) {
    activate_timer();
	BX_DEBUG(("activating timer...\n"));
    return;
    }
//BX_DEBUG(( "# not activating timer...\n");
//BX_DEBUG(( "#   allow_irq1 = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.allow_irq1);
//BX_DEBUG(( "#   outb       = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.outb);
//BX_DEBUG(( "#   clock_enab = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.kbd_clock_enabled);
//BX_DEBUG(( "#   out_buffer = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.kbd_output_buffer);
}

#if 0
  Boolean
bx_keyb_c::mouse_enQ_packet(Bit8u   b1, Bit8u   b2, Bit8u   b3)
{
  if ((BX_KEY_THIS s.mouse_internal_buffer.num_elements + 3) >= BX_MOUSE_BUFF_SIZE) {
    return(0); /* buffer doesn't have the space */
    }

//BX_INFO(("mouse: enQ_packet(%02x, %02x, %02x)\n",
//  (unsigned) b1, (unsigned) b2, (unsigned) b3));

  mouse_enQ(b1);
  mouse_enQ(b2);
  mouse_enQ(b3);
  return(1);
}


  void
bx_keyb_c::mouse_enQ(Bit8u   mouse_data)
{
  int tail;

  if (bx_dbg.mouse)
    BX_INFO(("mouse_enQ(%02x)\n", (unsigned) mouse_data));

  if (BX_KEY_THIS s.mouse_internal_buffer.num_elements >= BX_MOUSE_BUFF_SIZE) {
    BX_INFO(("mouse: internal mouse buffer full, ignoring mouse data.(%02x)\n",
      (unsigned) mouse_data));
    return;
    }
//BX_DEBUG(( "# mouse_enq() aux_clock_enabled = %u\n",
//  (unsigned) BX_KEY_THIS s.kbd_controller.aux_clock_enabled);

  /* enqueue mouse data in multibyte internal mouse buffer */
  tail = (BX_KEY_THIS s.mouse_internal_buffer.head + BX_KEY_THIS s.mouse_internal_buffer.num_elements) %
   BX_MOUSE_BUFF_SIZE;
  BX_KEY_THIS s.mouse_internal_buffer.buffer[tail] = mouse_data;
  BX_KEY_THIS s.mouse_internal_buffer.num_elements++;

  if (!BX_KEY_THIS s.kbd_controller.outb && BX_KEY_THIS s.kbd_controller.aux_clock_enabled) {
    activate_timer();
//fprintf(stderr, "# activating timer...\n");
    return;
    }
//fprintf(stderr, "# not activating timer...\n");
//fprintf(stderr, "#   allow_irq12= %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.allow_irq12);
//fprintf(stderr, "#   outb       = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.outb);
//fprintf(stderr, "#   clock_enab = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.aux_clock_enabled);
//fprintf(stderr, "#   out_buffer = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.aux_output_buffer);
}
#endif

  unsigned
bx_keyb_c::periodic( Bit32u   usec_delta )
{
  static int multiple=0;
  Bit8u   retval;

  UNUSED( usec_delta );

  if ( ++multiple==10)
  {
    multiple=0;
    bx_gui.handle_events();
  }
  retval = BX_KEY_THIS s.kbd_controller.irq1_requested;
  BX_KEY_THIS s.kbd_controller.irq1_requested = 0;

  if ( BX_KEY_THIS s.kbd_controller.timer_pending == 0 ) {
    return(retval);
    }

  if ( usec_delta >= BX_KEY_THIS s.kbd_controller.timer_pending ) {
    BX_KEY_THIS s.kbd_controller.timer_pending = 0;
    }
  else {
    BX_KEY_THIS s.kbd_controller.timer_pending -= usec_delta;
    return(retval);
    }

  if (BX_KEY_THIS s.kbd_controller.outb) {
    return(retval);
    }

  /* nothing in outb, look for possible data xfer from keyboard or mouse */
  if (BX_KEY_THIS s.kbd_internal_buffer.num_elements) {
//fprintf(stderr, "#   servicing keyboard code\n");
    if (bx_dbg.keyboard)
      BX_INFO(("KBD: service_keyboard: key in internal buffer waiting\n"));
    BX_KEY_THIS s.kbd_controller.kbd_output_buffer =
      BX_KEY_THIS s.kbd_internal_buffer.buffer[BX_KEY_THIS s.kbd_internal_buffer.head];
    BX_KEY_THIS s.kbd_controller.outb = 1;
    //BX_KEY_THIS s.kbd_controller.auxb = 0;
//fprintf(stderr, "# ___kbd::periodic kbd\n");
    BX_KEY_THIS s.kbd_internal_buffer.head = (BX_KEY_THIS s.kbd_internal_buffer.head + 1) %
      BX_KBD_ELEMENTS;
    BX_KEY_THIS s.kbd_internal_buffer.num_elements--;
    if (BX_KEY_THIS s.kbd_controller.allow_irq1)
      BX_KEY_THIS s.kbd_controller.irq1_requested = 1;
    }
#if 0
  else if (BX_KEY_THIS s.kbd_controller.aux_clock_enabled && BX_KEY_THIS s.mouse_internal_buffer.num_elements) {
//fprintf(stderr, "#   servicing mouse code\n");
    if (bx_dbg.mouse)
      BX_INFO(("KBD: service_keyboard: key in internal buffer waiting\n"));
    BX_KEY_THIS s.kbd_controller.aux_output_buffer =
      BX_KEY_THIS s.mouse_internal_buffer.buffer[BX_KEY_THIS s.mouse_internal_buffer.head];

    BX_KEY_THIS s.kbd_controller.outb = 1;
    BX_KEY_THIS s.kbd_controller.auxb = 1;
//fprintf(stderr, "# ___kbd:periodic aux\n");
    BX_KEY_THIS s.mouse_internal_buffer.head = (BX_KEY_THIS s.mouse_internal_buffer.head + 1) %
      BX_MOUSE_BUFF_SIZE;
    BX_KEY_THIS s.mouse_internal_buffer.num_elements--;
//fprintf(stderr, "#   allow12 = %u\n", (unsigned) BX_KEY_THIS s.kbd_controller.allow_irq12);
    if (BX_KEY_THIS s.kbd_controller.allow_irq12)
      BX_KEY_THIS s.kbd_controller.irq12_requested = 1;
    }
#endif
  else {
//fprintf(stderr, "#   servicing no code\n");
    if (bx_dbg.keyboard) {
      BX_INFO(("KBD: service_keyboard(): no keys waiting\n"));
      }
    }
  return(retval);
}




  void
bx_keyb_c::activate_timer(void)
{
  if (BX_KEY_THIS s.kbd_controller.timer_pending == 0) {
    BX_KEY_THIS s.kbd_controller.timer_pending = bx_options.keyboard_serial_delay;
    }
}



  void
bx_keyb_c::mouse_motion(int delta_x, int delta_y, unsigned button_state)
{
#if 0
  Bit8u   b1, b2, b3;

  // If mouse events are disabled on the GUI headerbar, don't
  // generate any mouse data
  if (bx_options.mouse_enabled==0)
    return;

  if ( BX_KEY_THIS s.mouse.enable==0 )
    return;

  // scale down the motion
  if ( (delta_x < -1) || (delta_x > 1) )
    delta_x /= 2;
  if ( (delta_y < -1) || (delta_y > 1) )
    delta_y /= 2;

#ifdef VERBOSE_KBD_DEBUG
  if (delta_x != 0 || delta_y != 0)
    BX_INFO(("[mouse] Dx=%d Dy=%d\n", delta_x, delta_y));
#endif  /* ifdef VERBOSE_KBD_DEBUG */

  b1 = (button_state & 0x0f) | 0x08; // bit3 always set

  BX_KEY_THIS s.mouse.button_status = button_state & 0x3;

  if ( (delta_x>=0) && (delta_x<=255) ) {
    b2 = delta_x;
    }
  else if ( delta_x > 255 ) {
    b2 = 0xff;
    }
  else if ( delta_x >= -256 ) {
    b2 = delta_x;
    b1 |= 0x10;
    }
  else {
    b2 = 0x00;
    b1 |= 0x10;
    }

  if ( (delta_y>=0) && (delta_y<=255) ) {
    b3 = delta_y;
    }
  else if ( delta_y > 255 ) {
    b3 = 0xff;
    }
  else if ( delta_y >= -256 ) {
    b3 = delta_y;
    b1 |= 0x20;
    }
  else {
    b3 = 0x00;
    b1 |= 0x20;
    }
  mouse_enQ_packet(b1, b2, b3);
#endif
}


  void
bx_keyb_c::put_scancode( unsigned char *code, int count )
{
  for ( int i = 0 ; i < count ; i++ ) {
    kbd_enQ( code[i] );
    }

  return;
}


  int
bx_keyb_c::SaveState( class state_file *fd )
{
  fd->write_check ("keyboard start");
  fd->write (&BX_KEY_THIS s, sizeof (BX_KEY_THIS s));
  fd->write_check ("keyboard end");
  return(0);
}


  int
bx_keyb_c::LoadState( class state_file *fd )
{
  fd->read_check ("keyboard start");
  fd->read (&BX_KEY_THIS s, sizeof (BX_KEY_THIS s));
  fd->read_check ("keyboard end");
  return(0);
}
#endif
